import math

# -------------------------
# Configuration Parameters
# -------------------------
SCALE = 10**10
MAX_RDIM = 255
MIN_OMEGA = 10**6
SLOTS_PER_INSTANCE = 4
SEED_VECTORS = 64
TARGET_VECTORS = 2**24  # 16,777,216
CTv_PUBKEY = "<YOUR_PUBLIC_KEY>"

# Fibonacci offsets for deterministic uniqueness
FIBS = [1,1,2,3,5,8,13,21,34,55,89,144,233,377,610]

# -------------------------
# Lattice Representation
# -------------------------
class LatticeVector:
    def __init__(self, idx, base_vals=(0.0,0.0,0.0,0.0)):
        self.idx = idx
        self.D_int = [int((v + idx/SEED_VECTORS) * SCALE) for v in base_vals]
        self.r_dim_scaled = int((idx/SEED_VECTORS) * MAX_RDIM)
        # Ω scaled with small Fibonacci offset for uniqueness
        self.omega_fixed = MIN_OMEGA + FIBS[idx % len(FIBS)] * 1000
    
    def pack_slots(self):
        return ''.join(f"{x:08x}" for x in self.D_int)

# -------------------------
# Script Generator
# -------------------------
def generate_fully_unique_covenant(pubkey=CTv_PUBKEY):
    script_lines = [f"{pubkey} OP_CHECKSIGVERIFY"]
    
    # Step 1: Generate unique seed vectors
    seed_vectors = [LatticeVector(i) for i in range(SEED_VECTORS)]
    for vec in seed_vectors:
        script_lines.append(f"0x{vec.pack_slots()} OP_HDGL_VEC_ADD OP_EQUALVERIFY")
        script_lines.append(f"{vec.r_dim_scaled} OP_ADD {vec.r_dim_scaled} OP_EQUALVERIFY")
        script_lines.append(f"{vec.omega_fixed} {vec.omega_fixed} {vec.omega_fixed} OP_WITHINVERIFY")
    
    # Step 2: Folding loop
    num_vectors = SEED_VECTORS
    fold_idx = 0
    while num_vectors < TARGET_VECTORS:
        fold_idx += 1
        fold_size = num_vectors
        num_vectors *= 2
        # Generate unique offsets per fold to ensure uniqueness
        r_delta = (fold_idx * 3) % MAX_RDIM
        omega_delta = MIN_OMEGA + fold_idx * 5000
        
        script_lines.append(f"OP_VEC_DUP {fold_size} OP_HDGL_VEC_ADD OP_EQUALVERIFY")
        script_lines.append(f"{r_delta} {r_delta} OP_HDGL_PARAM_ADD OP_EQUALVERIFY")
        script_lines.append(f"{omega_delta} {omega_delta} {omega_delta} OP_WITHINVERIFY")
    
    # Final check
    script_lines.append("OP_CHECKTEMPLATEVERIFY")
    return '\n'.join(script_lines)

# -------------------------
# Example Usage
# -------------------------
if __name__ == "__main__":
    covenant_script = generate_fully_unique_covenant()
    print("=== Generated Fully Folded 16,777,216-Vector HDGL Covenant ===")
    print(covenant_script)
